function New-PowerLabSwitch {
	param(
		[Parameter()]
		[string]$SwitchName = 'PowerLab',

		[Parameter()]
		[string]$SwitchType = 'External'
	)

	if (-not (Get-VmSwitch -Name $SwitchName -SwitchType $SwitchType -ErrorAction SilentlyContinue)) {
		$null = New-VMSwitch -Name $SwitchName -SwitchType $SwitchType
	} else {
		Write-Verbose -Message "The switch [$($SwitchName)] has already been created."
	}
}

function New-PowerLabVm {
	param(
		[Parameter(Mandatory)]
		[string]$Name,

		[Parameter()]
		[string]$Path = 'C:\PowerLab\VMs',

		[Parameter()]
		[string]$Memory = 4GB,

		[Parameter()]
		[string]$Switch = 'PowerLab',

		[Parameter()]
		[ValidateRange(1, 2)]
		[int]$Generation = 2,

		[Parameter()]
		[switch]$PassThru
	)

	if (-not (Get-Vm -Name $Name -ErrorAction SilentlyContinue)) {
		$null = New-VM -Name $Name -Path $Path -MemoryStartupBytes $Memory -Switch $Switch -Generation $Generation
	} else {
		Write-Verbose -Message "Maszyna wirtualna o nazwie [$($Name)] już istnieje."
	}
	if ($PassThru.IsPresent) {
		Get-VM -Name $Name
	}
}

function New-PowerLabVhd {
	param
	(
		[Parameter(Mandatory)]
		[string]$Name,

		[Parameter()]
		[string]$AttachToVm,

		[Parameter()]
		[ValidateRange(512MB, 1TB)]
		[int64]$Size = 50GB,

		[Parameter()]
		[ValidateSet('Dynamic', 'Fixed')]
		[string]$Sizing = 'Dynamic',

		[Parameter()]
		[string]$Path = 'C:\PowerLab\VHDs'
	)

	$vhdxFileName = "$Name.vhdx"
	$vhdxFilePath = Join-Path -Path $Path -ChildPath "$Name.vhdx"

	### Sprawdza, czy dysk VHD już istnieje; jeżeli nie, tworzy go
	if (-not (Test-Path -Path $vhdxFilePath -PathType Leaf)) {
		$params = @{
			SizeBytes = $Size
			Path      = $vhdxFilePath
		}
		if ($Sizing -eq 'Dynamic') {
			$params.Dynamic = $true
		} elseif ($Sizing -eq 'Fixed') {
			$params.Fixed = $true
		}

		New-VHD @params
		Write-Verbose -Message "Utworzono nowy dysk VHD w katalogu [$($vhdxFilePath)]"
	}

	### Przyłącza istniejący lub nowo utworzony dysk VHD do maszyny wirtualnej.
	if ($PSBoundParameters.ContainsKey('AttachToVm')) {
		if (-not ($vm = Get-VM -Name $AttachToVm -ErrorAction SilentlyContinue)) {
			Write-Warning -Message "Maszyna wirtualna o nazwie [$($AttachToVm)] nie istnieje. Nie można przyłączyć dysku VHD."
		} elseif (-not ($vm | Get-VMHardDiskDrive | Where-Object { $_.Path -eq $vhdxFilePath })) {
			$vm | Add-VMHardDiskDrive -Path $vhdxFilePath
			Write-Verbose -Message "Dysk VHDX [$($vhdxFilePath)] został przyłączony do maszyny wirtualnej [$($AttachToVM)]."
		} else {
			Write-Verbose -Message "Dysk VHDX [$($vhdxFilePath)] jest już przyłączony do maszyny wirtualnej [$($AttachToVM)]."
		}
	}
}

function Install-PowerLabOperatingSystem {
	param
	(
		[Parameter(Mandatory)]
		[string]$VmName,

		[Parameter()]
		[string]$OperatingSystem = 'Server 2016',

		[Parameter()]
		[ValidateSet('ServerStandardCore')]
		[string]$OperatingSystemEdition = 'ServerStandardCore',

		[Parameter()]
		[string]$DiskSize = 40GB,

		[Parameter()]
		[string]$VhdFormat = 'VHDX',

		[Parameter()]
		[string]$VhdType = 'Dynamic',

		[Parameter()]
		[string]$VhdPartitionStyle = 'GPT',

		[Parameter()]
		[string]$VhdBaseFolderPath = 'C:\PowerLab\VHDs',

		[Parameter()]
		[string]$IsoBaseFolderPath = 'C:\PowerLab\ISOs',

		[Parameter()]
		[string]$VhdPath
	)
	
	## Zakładamy, że w folderze mamy plik <VMName>.xml zawierający odpowiedzi dla instalacji nienadzorowanej
	$answerFile = Get-Item -Path "$PSScriptRoot\$VMName.xml"

	## Wywołanie skryptu w konwencji dot-source. Ponieważ w skrypcie znajduje się definicja funkcji, wykonanie tej operacji spowoduje jej udostępnienie
	. "$PSScriptRoot\Convert-WindowsImage.ps1"

	## Tutaj możemy dodać obsługę wielu systemów operacyjnych, wybierając odpowiedni obraz ISO w zależności od wybranej wersji systemu operacyjnego
	switch ($OperatingSystem) {
		'Server 2016' {
			$isoFilePath = "$IsoBaseFolderPath\en_windows_server_2016_x64_dvd_9718492.iso"
		}
		default {
			throw "Nierozpoznany parametr: [$_]"
		}
	}

	$convertParams = @{
		SourcePath        = $isoFilePath
		SizeBytes         = $DiskSize
		Edition           = $OperatingSystemEdition
		VHDFormat         = $VhdFormat
		VHDType           = $VhdType
		VHDPartitionStyle = 'GPT'
		UnattendPath      = $answerFile.FullName
	}
	if ($PSBoundParameters.ContainsKey('VhdPath')) {
		$convertParams.VHDPath = $VhdPath
	} else {
		$convertParams.VHDPath = "$VhdBaseFolderPath\$VMName.vhdx"
	}

	Convert-WindowsImage @convertParams

	$vm = Get-Vm -Name $VmName
	if (($vm | Get-VMHardDiskDrive).Path -ne $convertParams.VHDPath) {
		$vm | Add-VMHardDiskDrive -Path $convertParams.VHDPath
	}	
	$bootOrder = ($vm | Get-VMFirmware).Bootorder
	if ($bootOrder[0].BootType -ne 'Drive') {
		$vm | Set-VMFirmware -FirstBootDevice $vm.HardDrives[0]
	}
}